% fpi_comparison.m
% 
% comparison exercises of the FiPIt model
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %


% -- Clear Workspace -- %
clear
clc;
close all;
addpath('BFOR_Solution_Package')

% -- Take Symbolic Derivatives -- %
[f, Pmat, vars, THETA, steady] = bfor_init_vars_equilibriums_fipit();
[derivs] = bfor_endogenous_ms_solver_derivatives_analytic(f, Pmat, vars);

% -- Parameters -- %
GLOBALS.ny = 16;
GLOBALS.nx = 5;
GLOBALS.ne = 3;
GLOBALS.ns = 2;
GLOBALS.nY = 21;
GLOBALS.simlength = 100;
GLOBALS.simburn   = 1;
GLOBALS.obsindex  = 1:21;

% -- Parameters and Solution -- %
p01_0    = 0.95;
p10_0    = 0.95;
ggamma00 = log(p01_0/(1-p01_0));
ggamma10 = log(p10_0/(1-p10_0));
ggamma01 = 50;
ggamma11 = 50;
GLOBALS.parameters.ggamma00 = ggamma00;
GLOBALS.parameters.ggamma10 = ggamma10;
GLOBALS.parameters.ggamma01 = ggamma01;
GLOBALS.parameters.ggamma11 = ggamma11;

ssigma      = 2.0;
oomega      = 1.8461;
bbeta       = 0.92;
iiota       = 2.75;
pphi        = 0.2579;
ddelta      = 0.08800443;
aalpha      = 0.5924043;
eeta        = 0.102229;
ggamma      = 1-aalpha-eeta;
ttau        = 0.168615385890;
astar       = 6.982;
kkappa      = 0.2;
rbar        = 1.0857;
pbar        = 1.0278;
abar        = 1;
rrhor       = 0.5551;
rrhop       = 0.7374;
rrhoa       = 0.5551;
ssigmaa     = 0.0100185;
ssigmap     = 0.023235697402196;
ssigmar     = 0.0170303;
ssigmaar    = -0.00488483;
varphi_1    = 0;
varphi_2    = 1;
nu_1        = 0;
nu_2        = 1;
% Paramter Vector
nTHETA   = eval(THETA);


[xss, yss, sss, Pss, H, G, H2, G2, retcode] = ...
            bfor_endogenous_ms_solver_main(f, Pmat, vars, derivs, steady, THETA, nTHETA);

meanstable  = fpi_compare_stochasticss(GLOBALS,xss,yss,H,G,H2,G2,1:21);
bstar_stochss = meanstable(13,1);
p01_ss  = exp(ggamma00-ggamma01*bstar_stochss)/(1+exp(ggamma00-ggamma01*bstar_stochss));


mu_stochss = meanstable(11,1);
p10_ss  = exp(ggamma10-ggamma11*mu_stochss)/(1+exp(ggamma10-ggamma11*mu_stochss));


% -- Simulation -- %
[simdataout,regimesout] = fpi_compare_simdata(GLOBALS,THETA,Pss,xss,yss,H,G,H2,G2);

% -- Simulation Output -- %
% predetermined
k       = exp(simdataout(:,1));
b       = simdataout(:,2);
a       = exp(simdataout(:,3));
p       = exp(simdataout(:,4));
rstar   = exp(simdataout(:,5));
% nonpredetermined
c       = exp(simdataout(:,6));
l       = exp(simdataout(:,7));
v       = exp(simdataout(:,8));
q       = exp(simdataout(:,9));
w       = exp(simdataout(:,10));
mu      = simdataout(:,11);
lam   	= exp(simdataout(:,12));
bstar	= simdataout(:,13);
d       = exp(simdataout(:,14));
r       = exp(simdataout(:,15));
gdp     = exp(simdataout(:,16));
inv     = exp(simdataout(:,17));
wc      = exp(simdataout(:,18));
nx_gdp  = simdataout(:,19);
lev     = simdataout(:,20);
b_gdp   = simdataout(:,21);

% -- Table -- %
tab     = [gdp c inv nx_gdp k b_gdp q lev v wc];
means   = mean(tab);
stds    = std(tab);
correls = corr(tab);
autocor = corr(tab(2:end,:),tab(1:end-1,:));

% -- Compute Durations -- %
temp = double(regimesout==2);
for tt = 1:length(temp)-1
if temp(tt) > 0 && temp(tt+1) == 1
    temp(tt+1) = temp(tt+1) + temp(tt);
    temp(tt) = 0;
end
end
temp = temp(temp>0);
duration2 = mean(temp);

temp = (regimesout==2).*(mu>0);
for tt = 1:length(temp)-1
if temp(tt) > 0 && temp(tt+1) == 1
    temp(tt+1) = temp(tt+1) + temp(tt);
    temp(tt) = 0;
end
end
temp = temp(temp>0);
durationmu = mean(temp);    


% -- Output -- %
output  = [ggamma00; ggamma01; ggamma10; ggamma11;    
            p01_0; p10_0; bstar_stochss; p01_ss; mu_stochss; p10_ss;
            means'; 
            100*stds(1:3)'./means(1:3)'; 100*stds(4); 100*stds(5)./means(5); 100*stds(6); 100*stds(7)'./means(7)'; 100*stds(8); 100*stds(9:10)'./means(9:10)';
            correls(:,1); ...
            diag(autocor); ...
            100*sum(regimesout==2)/length(regimesout);...
            mean(duration2);
            100*sum(mu(regimesout==2)>0)/length(regimesout)
            mean(durationmu); NaN;
            100*std(diff(log(tab(:,[1 2 3]))))'];
disp(output)

% -- Density Plot -- %
figure(201)
subplot(2,2,1)
    ksdensity(b);
    xlim([-100 50]);
subplot(2,2,2)
    ksdensity(k);
    xlim([600 950])
subplot(2,1,2)
    ksdensity([b k]);
    xlim([-100 50])
    ylim([600 950])
    print('-dpdf','../Figures and Tables/Figure_C1','-fillpage')


% -- Decision Rules -- %
kmin=700;
kmax=800;
bmin=-100;
bmax=-0;


bgrid = bmin:bmax;
kgrid = log(kmin:kmax);
brule{1} = NaN(length(bgrid),length(kgrid));
brule{2} = NaN(length(bgrid),length(kgrid));
krule{1} = NaN(length(bgrid),length(kgrid));
krule{2} = NaN(length(bgrid),length(kgrid));
qrule{1} = NaN(length(bgrid),length(kgrid));
qrule{2} = NaN(length(bgrid),length(kgrid));
bstarrule{1} = NaN(length(bgrid),length(kgrid));
bstarrule{2} = NaN(length(bgrid),length(kgrid));

for ib = 1:length(bgrid)
    for ik = 1:length(kgrid)
        state = [kgrid(ik)-xss(1);bgrid(ib)-xss(2);0;0;0;zeros(3,1);1];
        
        % non-binding
        x = xss + H(:,:,1)*state + 1/2*H2(:,:,1)*kron(state,state);
        y = yss + G(:,:,1)*state + 1/2*G2(:,:,1)*kron(state,state);
        krule{1}(ib,ik) = exp(x(1));
        brule{1}(ib,ik) = x(2);
        qrule{1}(ib,ik) = exp(y(4));
        bstarrule{1}(ib,ik) = y(8);

        % binding
        x = xss + H(:,:,2)*state + 1/2*H2(:,:,2)*kron(state,state);
        y = yss + G(:,:,2)*state + 1/2*G2(:,:,2)*kron(state,state);
        krule{2}(ib,ik) = exp(x(1));
        brule{2}(ib,ik) = x(2);
        qrule{2}(ib,ik) = exp(y(4));
        bstarrule{2}(ib,ik) = y(8);
    end
end



figure(202)
subplot(3,2,1)
    surf(bgrid,exp(kgrid),krule{1}','EdgeColor','r','FaceColor','r');
    ylabel('capital')
    xlabel('bond')
    axis tight
    ylim([700 800])
    xlim([-60 -20])
    title('Capital: Non-binding');   
subplot(3,2,2)
    surf(bgrid,exp(kgrid),krule{2}','EdgeColor','b','FaceColor','b')
    ylabel('capital')
    xlabel('bond')
    axis tight
    ylim([700 800])
    xlim([-100 -50])
    title('Capital: Binding');  
subplot(3,2,3)
    surf(bgrid,exp(kgrid),brule{1}','EdgeColor','r','FaceColor','r');
    ylabel('capital')
    xlabel('bond')
    axis tight
    ylim([700 800])
    xlim([-60 -20])
    title('Bonds: Non-binding');   
subplot(3,2,4)
    surf(bgrid,exp(kgrid),brule{2}','EdgeColor','b','FaceColor','b')
    ylabel('capital')
    xlabel('bond')
    ylim([700 800])
    xlim([-100 -50])
    title('Bonds: Binding');  
subplot(3,2,5)
    surf(bgrid,exp(kgrid),bstarrule{1}','EdgeColor','r','FaceColor','r')
    ylabel('capital')
    xlabel('bond')
    ylim([700 800])
    xlim([-60 -20])
    title('Borrowing Cushion: Non-binding'); 
subplot(3,2,6)
    surf(bgrid,exp(kgrid),bstarrule{2}','EdgeColor','b','FaceColor','b')
    ylabel('capital')
    xlabel('bond')
    ylim([700 800])
    xlim([-100 -50])
    title('Borrowing Cushion: Binding');   
print('-dpdf','../Figures and Tables/Figure_C2','-fillpage')

